#include <stdint.h>
#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <math.h>
#include <string.h>

#define ARRAY_DEFAULT_CAPACITY 8

typedef struct {
    uint64_t* arr;
    size_t length;
    size_t capacity;
} uint64_array_t;

uint64_array_t* uint64_array_create() {
    uint64_array_t* arr = malloc(sizeof(uint64_array_t));
    if (arr == NULL) {
        return NULL;
    }
    arr->capacity = ARRAY_DEFAULT_CAPACITY;
    arr->length = 0;
    arr->arr = malloc(ARRAY_DEFAULT_CAPACITY * sizeof(uint64_t));
    if (arr->arr == NULL) {
        free(arr);
        return NULL;
    }
    return arr;
} 

void uint64_array_free(uint64_array_t* arr) {
    free(arr->arr);
    free(arr);
}

uint8_t uint64_array_capacity_increase(uint64_array_t* arr) {
    arr->capacity *= 2;
    arr->arr = realloc(arr->arr, arr->capacity * sizeof(uint64_t));
    if (arr->arr == NULL) {
        return 1;
    }
    return 0;
}

uint8_t uint64_array_append(uint64_array_t* arr, uint64_t value) {
    if (arr->length == arr->capacity) {
        if (uint64_array_capacity_increase(arr)) {
            return 1;
        }
    }

    arr->arr[arr->length] = value;
    arr->length += 1;

    return 0;
}

typedef struct {
    uint64_t u;
    uint64_t n;
} args_t;

void parse_args(args_t* args, int argc, char* argv[]) {
    args->u = 0;
    args->n = 0;

    for (size_t i=1; i < argc; i++) {
        if (strcmp(argv[i], "-n") == 0 && i + 1 < argc) {
            args->n = strtol(argv[++i], NULL, 10);
        } else if (strcmp(argv[i], "-u") == 0 && i + 1 < argc) {
            args->u = strtol(argv[++i], NULL, 10);
        } else if (strcmp(argv[i], "--help") == 0) {
            fprintf(stderr, "Usage: sieve\nParameters:\n\t-n NUMBER: Print NUMBER prime numbers\n\t-u NUMBER: Print prime numbers until NUMBER is reached\n\t--help: Show this help\n");
            exit(0);
        }
    }
}

void handle_sigint(int sig) {
    fflush(stdout);
    exit(0);
}

int main(int argc, char* argv[]) {
    args_t args;
    parse_args(&args, argc, argv);

    signal(SIGINT, handle_sigint);
    
    uint64_array_t* arr = uint64_array_create();
    if (arr == NULL) {
        perror("Error while allocating space for array");
        return 1;
    }
    uint64_t i = 2;
    while (1) {
        int limit = (int) sqrt((double) i) + 1;
        for (size_t k=0; k < arr->length; k++) {
            int p = arr->arr[k];
            if (p > limit) break;
            if (i % p == 0) {
                goto incr;
            }
        }

        printf("%llu\n", i);
        if (uint64_array_append(arr, i)) {
            goto error;
        }
        if (args.n > 0 && arr->length == args.n) {
            break;
        }

        incr: i += 1;
        if (args.u > 0 && i > args.u) {
            break;
        }
    }
    end:
    uint64_array_free(arr);
    fflush(stdout);
    return 0;

    error:
    uint64_array_free(arr);
    perror("Error");
    return 1;
}