import pandas as pd
import re

from datetime import datetime, date, timedelta

BC_START_DATE = date(2025, 3, 6)
FGR_AVERAGE_PROCESSING_TIME = timedelta(days=5)
PRICE = 2988.00
FILENAME = "Transaktionen.csv"

TODAY = date.today() - FGR_AVERAGE_PROCESSING_TIME
BC_END_DATE = BC_START_DATE + timedelta(days=365)

def parse_journey_date(value):
    match = re.search(r"Bahnreise am (\d{1,2}\.\d{1,2}\.\d{4})", str(value))
    if match:
        return datetime.strptime(match.groups()[0], "%d.%m.%Y").date()

df = pd.read_csv(
    FILENAME,
    converters={
        "Betrag": lambda x: float(x.replace(",", ".")),
        "Buchungsdatum": lambda x: datetime.fromisoformat(x).date()
    }
)

df["Reisedatum"] = df["Verwendungszweck"].map(parse_journey_date)

df = df[
    (df["Zahlungsempfänger"] == "Deutsche Bahn") &
    (df["Verwendungszweck"].str.contains("Fahrgastrechtsfall", case=False)) & 
    (df["Betrag"] == 10) & 
    (df["Buchungsdatum"] > BC_START_DATE) &
    (df["Reisedatum"] >= BC_START_DATE)
]

current_amount = df["Betrag"].sum()
bc_use_percentage= ((TODAY - BC_START_DATE).days / 365)
max_amount = bc_use_percentage * PRICE * 0.25

effective_price = PRICE - (current_amount * (1 / bc_use_percentage))

print("Fahrgastrechte: {:.2f}€ von {:.2f}€ ({:.0f}%)".format(
    current_amount,
    max_amount,
    (current_amount / max_amount) * 100
))

print("Effektiver Preis der BC100 wird statt {:.2f}€ {:.2f}€ sein ({:.0f}% Rabatt)".format(
    PRICE,
    effective_price,
    (1 - (effective_price / PRICE)) * 100
))